xquery version "1.0-ml";

module namespace update="http://www.marklogic.com/ns/nwalsh/twitter/update";

(: update provides higher-level functions for getting status messages from
 : a Twitter API server.
 :
 : @author Norman Walsh, norman.walsh@marklogic.com
 : @date 28 Aug 2009
 :)

import module namespace twit="http://www.marklogic.com/ns/nwalsh/twitter"
       at "/modules/twitter.xqy";

import module namespace twitproc="http://www.marklogic.com/ns/nwalsh/twitter/process"
       at "/modules/twitproc.xqy";

declare default function namespace "http://www.w3.org/2005/xpath-functions";

declare namespace t="http://www.marklogic.com/ns/nwalsh/twitter/tweets";
declare namespace xh="xdmp:http";
declare namespace h="http://www.w3.org/1999/xhtml";

declare option xdmp:mapping "false"; 

(: update:get-my-most-recent returns the ID of the highest-numbered status
 : message in the database that was authored by the specified user.
 :
 : @param $account identifies the MBB user.
 : @return The xs:decimal value of the highest numbered status message
 :         authored by the user identified by $account, or the empty sequence
 :         if there are not messages by them in the database.
 :)
declare function update:get-my-most-recent($account as element(t:account))
        as xs:decimal?
{
  let $tweet-collection
    := concat("http://www.marklogic.com/collections/tweets/",
              $account/t:service, "/", $account/t:screen_name)

  let $service-q := cts:element-value-query(xs:QName("t:service"),
                                           string($account/t:service),
                                           ("exact"))

  let $author-q := cts:element-value-query(xs:QName("t:screen_name"),
                                           string($account/t:screen_name),
                                           ("exact"))

  let $my-q 
    := cts:and-query(($service-q, $author-q), "unordered")

  return
    xs:decimal(max(cts:search(collection(), $my-q)/t:status/t:id))
};

(: update:get-friends-most-recent returns the ID of the highest-numbered status
 : message in the database that was downloaded by the specified user but
 : authored by someone else.
 :
 : @param $account identifies the MBB user.
 : @return The xs:decimal value of the highest numbered status message
 :         downloaded on behalf of this user, but authored by someone else,
 :         or the empty sequence if there are no such messages.
 :)
declare function update:get-friends-most-recent($account as element(t:account))
        as xs:decimal?
{
  let $tweet-collection
    := concat("http://www.marklogic.com/collections/tweets/",
              $account/t:service, "/", $account/t:screen_name)

  let $service-q := cts:element-value-query(xs:QName("t:service"),
                                           string($account/t:service),
                                           ("exact"))

  let $author-q := cts:element-value-query(xs:QName("t:screen_name"),
                                           string($account/t:screen_name),
                                           ("exact"))

  let $login-q  := cts:element-value-query(xs:QName("t:login"),
                                           string($account/t:screen_name),
                                           ("exact"))

  let $by-me := cts:and-query(($service-q, $login-q), ("unordered"))

  let $other-q 
    := cts:and-not-query($by-me, $author-q)

  return
    xs:decimal(max(cts:search(collection(), $other-q)/t:status/t:id))
};

(: update:get-tweets gets all the new status messages for a particular user.
 :
 : This function downloads all of the new messages on the user's timeline
 : and the user's friend's timeline and inserts them into the database.
 :
 : @param $account identifies the MBB user.
 : @return Zero or more html:li elements, one for each status message
 :         inserted into the database.
 :)
declare function update:get-tweets($account as element(t:account)) 
        as element(h:li)*
{
  let $d0 := xdmp:log(concat("Getting tweets for ", $account/t:screen_name, 
                             " on ", $account/t:service))

  let $my-high-tweet := update:get-my-most-recent($account)
  let $friends-high-tweet := update:get-friends-most-recent($account)

  let $d1 := xdmp:log(concat("My highest tweet: ", $my-high-tweet))
  let $d2 := xdmp:log(concat("Highest fr tweet: ", $friends-high-tweet))

  let $rate-limit := twit:account.rate-limit-status($account)

  return
    if ((not(empty($my-high-tweet))
         and not(empty($friends-high-tweet))
         and $rate-limit > 20) 
        or ($rate-limit > 50))
    then
      let $my-tweets
	:= twit:statuses.user-timeline($account, $my-high-tweet)
      let $friends-tweets
	:= twit:statuses.friends-timeline($account, $friends-high-tweet)

      (: Friends-timeline can return *my* tweets, which causes a conflicting
         updates exception, so filter mine out. :)
      let $other-tweets as element(statuses)
	:= <statuses>
             { $friends-tweets/@* }
             { $friends-tweets/status[user/screen_name != $account/t:screen_name] }
	   </statuses>

      (: FIXME: On at least one occasion, I saw a conflicting updates exception
         that I think was caused by a duplicate tweet in either my or my friends
         timeline. I suspect that a database update on Twitter caused the same
         message to appear at the end of one page and the beginning of another.
         I could (probably should) do a little work here to filter out duplicates.
         But it's rare and I'm feeling lazy. :)

      return
        (if (not($my-tweets/*) and not($other-tweets/*))
	 then
	   <li xmlns="http://www.w3.org/1999/xhtml">No new statuses.</li>
	 else
	   (),
	 twitproc:update-statuses($account, $my-tweets, true()),
         twitproc:update-statuses($account, $other-tweets, false()))
    else
      <li xmlns="http://www.w3.org/1999/xhtml">Skipped: too near rate limit</li>
};
