xquery version "1.0-ml";

module namespace twitter="http://www.marklogic.com/ns/nwalsh/twitter";

(: twitter implements the Twitter API calls used by MBB.
 :
 : @author Norman Walsh, norman.walsh@marklogic.com
 : @date 28 Aug 2009
 :)

import module namespace acct="http://www.marklogic.com/ns/nwalsh/twitter/accounts"
       at "/modules/accounts.xqy";

declare default function namespace "http://www.w3.org/2005/xpath-functions";

declare namespace t="http://www.marklogic.com/ns/nwalsh/twitter/tweets";
declare namespace xh="xdmp:http";

declare option xdmp:mapping "false"; 

(: $MAX-PAGES is the largest number of pages that will be requested
 : from a user's or friends' timeline. This prevents a service which
 : allows an unbounded number of status messages to be requested from
 : causing us to blow the stack.
 :)
declare variable $MAX-PAGES as xs:decimal := 20;

(: ============================================================ :)
(: twitter:account.rate-limit-status returns the number of requests
 : remaining in the current hour.
 :
 : @param $account identifies the user.
 : @return The rate limit status, or 0 if an error occurs.
 :)
declare function twitter:account.rate-limit-status($account as element(t:account))
        as xs:decimal {
  let $uri := concat($account/t:api-uri,"/account/rate_limit_status.xml")
  let $hash := twitter:_api($account, $uri)
  let $d1 := xdmp:log(concat("Rate limit for ", $account/t:screen_name,
                             " on ", $account/t:service,
			     ": ", string($hash/remaining-hits)))
  return
    if ($hash/remaining-hits) 
    then
      xs:decimal($hash/remaining-hits)
    else
      (0,xdmp:log($hash))
};

(: ============================================================ :)
(: twitter:statuses.show returns a particular status message.
 :
 : @param $account identifies the user.
 : @param $id identifies the particular status message requested.
 : @return The status message as an element(status) or,
 :         if an error occurs, the HTTP error code as an
 :         element(t:status).
 :)
declare function twitter:statuses.show($account as element(t:account),
                                       $id as xs:decimal)
        as element()?
{
  let $uri := concat($account/t:api-uri,"/statuses/show/",$id,".xml")
  return
    twitter:_api($account, $uri)
};

(: ============================================================ :)
(: twitter:statuses.user-timeline returns all of the available
 : messages from the user's timeline.
 :
 : Unlike the underlying API which returns at most one page,
 : this function always collects all the available pages and
 : returns them in one document. If multiple pages are requested,
 : they are combined together into a single page.
 :
 : @param $account identifies the user.
 : @return The status messages as an element(statuses) or,
 :         if an error occurs, the HTTP error code as an
 :         element(t:status).
 :)
declare function twitter:statuses.user-timeline($account as element(t:account))
        as element()?
{
  twitter:statuses.user-timeline($account, ())
};

(: twitter:statuses.user-timeline returns all of the available
 : messages from the user's timeline since a particular ID.
 :
 : Unlike the underlying API which returns at most one page,
 : this function always collects all the available pages and
 : returns them in one document. If multiple pages are requested,
 : they are combined together into a single page.
 :
 : @param $account identifies the user.
 : @param $since-id identifies the oldest status to consider. If it is
 :        not specified, all available statuses are returned.
 : @return The status messages as an element(statuses) or,
 :         if an error occurs, the HTTP error code as an
 :         element(t:status).
 :)
declare function twitter:statuses.user-timeline($account as element(t:account),
                                                $since-id as xs:decimal?)
        as element()?
{
    twitter:_statuses.user-timeline($account, 200, 1, $since-id)
};

(: twitter:_statuses.user-timeline is an internal function that retrieves
 : and collates all of the pages of statuses together.
 :)
declare function twitter:_statuses.user-timeline($account as element(t:account),
			                         $count as xs:decimal,
			                         $page as xs:decimal,
			                         $since as xs:decimal?)
        as element()?
{
  let $statuses := twitter:_get-statuses-page($account, $count, $page, $since)
  let $stats := $statuses/status
  return
    <statuses>
      { $statuses/@* }
      { $stats }
      { (: don't blow the stack... :) }
      { if (not(empty($stats)) and $page < $MAX-PAGES)
        then
	  twitter:_statuses.user-timeline($account, $count, $page+1, $since)/*
	else
	  ()
      }
      </statuses>
};

(: twitter:_get-statuses-page is an internal function that retrieves
 : a single page of statuses from the user's timeline.
 :)
declare function twitter:_get-statuses-page($account as element(t:account),
			                    $count as xs:decimal,
			                    $page as xs:decimal,
					    $since as xs:decimal?)
        as element()?
{
  let $uri := concat($account/t:api-uri,"/statuses/user_timeline.xml")
  let $qparam := concat("count=", $count,
                        "&amp;screen_name=", $account/t:screen_name,
			"&amp;page=", $page,
		        if (not(empty($since)))
			then
			  concat("&amp;since_id=", $since)
			else
			  "")
  return
    twitter:_api($account, concat($uri, "?", $qparam))
};

(: ============================================================ :)
(: twitter:statuses.friends-timeline returns all of the available
 : messages from the user's friend's timeline.
 :
 : Unlike the underlying API which returns at most one page,
 : this function always collects all the available pages and
 : returns them in one document. If multiple pages are requested,
 : they are combined together into a single page.
 :
 : @param $account identifies the user.
 : @return The status messages as an element(statuses) or,
 :         if an error occurs, the HTTP error code as an
 :         element(t:status).
 :)
declare function twitter:statuses.friends-timeline($account as element(t:account))
        as element()?
{
  twitter:statuses.friends-timeline($account, ())
};

(: ============================================================ :)
(: twitter:statuses.friends-timeline returns all of the available
 : messages from the user's friend's timeline.
 :
 : Unlike the underlying API which returns at most one page,
 : this function always collects all the available pages and
 : returns them in one document. If multiple pages are requested,
 : they are combined together into a single page.
 :
 : @param $account identifies the user.
 : @param $since-id identifies the oldest status to consider. If it is
 :        not specified, all available statuses are returned.
 : @return The status messages as an element(statuses) or,
 :         if an error occurs, the HTTP error code as an
 :         element(t:status).
 :)
declare function twitter:statuses.friends-timeline($account as element(t:account),
                                                   $since-id as xs:decimal?)
        as element()?
{
    twitter:_statuses.friends-timeline($account, 200, 1, $since-id)
};

(: twitter:_statuses.friends-timeline is an internal function that retrieves
 : and collates all of the pages of statuses together.
 :)
declare function twitter:_statuses.friends-timeline($account as element(t:account),
                                                    $count as xs:decimal,
                                                    $page as xs:decimal,
                                                    $since as xs:decimal?)
        as element()?
{
  let $statuses := twitter:_get-friends-page($account, $count, $page, $since)
  let $stats := $statuses/status
  return
    <statuses>
      { $statuses/@* }
      { $stats }
      { if (not(empty($stats)) and $page < $MAX-PAGES)
        then
	  twitter:_statuses.friends-timeline($account, $count, $page+1, $since)/*
	else
	  ()
      }
    </statuses>
};

(: twitter:_get-friends-page is an internal function that retrieves
 : a single page of statuses from the user's friend's timeline.
 :)
declare function twitter:_get-friends-page($account as element(t:account),
			                   $count as xs:decimal,
			                   $page as xs:decimal,
					   $since as xs:decimal?)
        as element()?
{
  let $uri := concat($account/t:api-uri,"/statuses/friends_timeline.xml")
  let $qparam := concat("count=", $count,
			"&amp;page=", $page,
		        if (not(empty($since)))
			then
			  concat("&amp;since_id=", $since)
			else
			  "")
  return
    twitter:_api($account, concat($uri, "?", $qparam))
};

(: ============================================================ :)
(: twitter:_api is an internal function that retrieves a URI
 : using the authentication credentials of the account.
 : 
 : It returns either the document, of t:error containing the
 : HTTP error code if an error occurred.
 :)
declare function twitter:_api($account as element(t:account),
			      $uri as xs:string)
        as element()?
{
  try {
    let $opts
      := <options xmlns="xdmp:http">
           {acct:get-credentials($account)}
	 </options>
    let $d1 := xdmp:log(concat("Getting ", $uri))
    let $result := xdmp:http-get($uri, $opts)
    let $d2 := xdmp:log(concat("HTTP returned ", $result[1]/xh:code))
    return
      if ($result[1]/xh:code = 200)
      then
        $result[2]/*
      else
        (xdmp:log($result[1]),
         <t:error>{string($result[1]/xh:code)}</t:error>)
  } catch ($ex) {
    xdmp:log($ex)
  }
};
