xquery version "1.0-ml";

module namespace accounts="http://www.marklogic.com/ns/nwalsh/twitter/accounts";

(: accounts implements a few convenience methods for accessing user account data.
 :
 : @author Norman Walsh, norman.walsh@marklogic.com
 : @date 28 Aug 2009
 :)

declare default function namespace "http://www.w3.org/2005/xpath-functions";

declare namespace t="http://www.marklogic.com/ns/nwalsh/twitter/tweets";
declare namespace xh="xdmp:http";

declare variable $ENOSUCHACCT as xs:QName := xs:QName("t:ENOSUCHACCT");

(: ============================================================ :)
(: accounts:services returns the names of all the services 
 : configured for this database.
 :
 : @return A list of the services configured.
 :)
declare function accounts:services() as xs:string* {
  distinct-values(for $account in /t:account return $account/t:service)
};

(: ============================================================ :)
(: accounts:logins returns the screen_names of all the users 
 : associated with a particular service.
 :
 : @return A list of the screen_names configured.
 :)
declare function accounts:logins($service as xs:string) as xs:string* {
  distinct-values(for $account in /t:account[t:service = $service]
                  return $account/t:screen_name)
};

(: ============================================================ :)
(: accounts:logins returns the t:account associated with the
 : user who was logged in when this status message was
 : retreived.
 :
 : @return A t:account for the user who was logged in to get this tweet.
 :)
declare function accounts:get-login($tweet as element(t:status))
        as element(t:account) {
  let $uri := xdmp:node-uri($tweet)
  let $service := string($tweet/t:service)
  let $name := string($tweet/t:login)
  return
    accounts:get-login($service, $name)
};

(: ============================================================ :)
(: accounts:get-login returns the t:account associated with the
 : service and screen_name specified.
 :
 : @return A t:account for the user.
 : @throws $ENOSUCHACCT if no such service/screen_name combination
 :         is configured.
 :)
declare function accounts:get-login($service as xs:string, $name as xs:string)
        as element(t:account)
{
  let $account := /t:account[t:service = $service and t:screen_name = $name]
  return
    if (count($account) != 1)
    then
      error($ENOSUCHACCT,
            concat("Cannot find login for ", $name, " on ", $service))
    else
      $account
};

(: ============================================================ :)
(: accounts:get-credentials returns an xdmp:http authentication
 : element for the specified account.
 :
 : @return A {xdmp:http}authentication element
 :)
declare function accounts:get-credentials($account as element(t:account))
        as element(xh:authentication)?
{
  <authentication xmlns="xdmp:http" method="basic">
    <username>{string($account/t:screen_name)}</username>
    <password>{string($account/t:password)}</password>
  </authentication>
};
